import SwiftUI

struct PrivacySettingsView: View {
    @StateObject private var disclaimerManager = DisclaimerManager()
    @State private var showingDataDeletionConfirmation = false
    @State private var showingDisclaimerReset = false
    @State private var showingDataExport = false
    @Environment(\.dismiss) private var dismiss
    
    var body: some View {
        NavigationStack {
            List {
                Section {
                    DisclaimerStatusRow()
                    
                    Button("Review Medical Disclaimer") {
                        showingDisclaimerReset = true
                    }
                    .foregroundColor(.blue)
                } header: {
                    Text("Medical Disclaimer")
                } footer: {
                    Text("Medical disclaimer must be accepted to use the app")
                }
                
                Section {
                    Button("Export My Data") {
                        showingDataExport = true
                    }
                    .foregroundColor(.blue)
                    
                    Button("Delete All Data") {
                        showingDataDeletionConfirmation = true
                    }
                    .foregroundColor(.red)
                } header: {
                    Text("Data Management")
                } footer: {
                    Text("You have full control over your personal data")
                }
                
                Section {
                    DataStorageInfoView()
                } header: {
                    Text("Privacy Information")
                }
                
                Section {
                    ComplianceStatusView()
                } header: {
                    Text("App Store Compliance")
                }
            }
            .navigationTitle("Privacy & Data")
            .navigationBarTitleDisplayMode(.large)
            .toolbar {
                ToolbarItem(placement: .cancellationAction) {
                    Button("Done") {
                        dismiss()
                    }
                }
            }
        }
        .confirmationDialog(
            "Delete All Data",
            isPresented: $showingDataDeletionConfirmation,
            titleVisibility: .visible
        ) {
            Button("Delete Everything", role: .destructive) {
                deleteAllUserData()
            }
            Button("Cancel", role: .cancel) { }
        } message: {
            Text("This will permanently delete all your health data, survey responses, and app preferences. This action cannot be undone.")
        }
        .sheet(isPresented: $showingDisclaimerReset) {
            DisclaimerView(hasAcceptedDisclaimer: $disclaimerManager.hasAcceptedDisclaimer)
        }
        .sheet(isPresented: $showingDataExport) {
            DataExportView()
        }
    }
    
    private func deleteAllUserData() {
        // Clear all UserDefaults related to the app
        let domain = Bundle.main.bundleIdentifier!
        UserDefaults.standard.removePersistentDomain(forName: domain)
        UserDefaults.standard.synchronize()
        
        // Clear Core Data if used
        // clearCoreDataStore()
        
        // Clear any cached files
        clearCachedData()
        
        // Reset disclaimer manager
        disclaimerManager.resetDisclaimer()
        
        // Show confirmation
        // Could add a success toast or alert here
    }
    
    private func clearCachedData() {
        // Clear any temporary files or caches
        let cacheURL = FileManager.default.urls(for: .cachesDirectory, in: .userDomainMask).first!
        try? FileManager.default.removeItem(at: cacheURL)
    }
}

struct DisclaimerStatusRow: View {
    @StateObject private var disclaimerManager = DisclaimerManager()
    
    var body: some View {
        HStack {
            Image(systemName: disclaimerManager.hasAcceptedDisclaimer ? "checkmark.circle.fill" : "exclamationmark.triangle.fill")
                .foregroundColor(disclaimerManager.hasAcceptedDisclaimer ? .green : .orange)
            
            VStack(alignment: .leading, spacing: 2) {
                Text("Medical Disclaimer")
                    .font(.headline)
                
                if disclaimerManager.hasAcceptedDisclaimer {
                    if let acceptanceDate = disclaimerManager.getAcceptanceDate() {
                        Text("Accepted: \(acceptanceDate.formatted(date: .abbreviated, time: .omitted))")
                            .font(.caption)
                            .foregroundColor(.secondary)
                    }
                } else {
                    Text("Not Accepted")
                        .font(.caption)
                        .foregroundColor(.red)
                }
            }
            
            Spacer()
        }
    }
}

struct DataStorageInfoView: View {
    var body: some View {
        VStack(alignment: .leading, spacing: 12) {
            InfoRow(
                icon: "iphone",
                title: "Local Storage Only",
                description: "All data is stored securely on your device"
            )
            
            InfoRow(
                icon: "lock.shield",
                title: "No Data Transmission",
                description: "No personal information is sent to external servers"
            )
            
            InfoRow(
                icon: "person.crop.circle.badge.checkmark",
                title: "Full Control",
                description: "You can export or delete your data at any time"
            )
            
            InfoRow(
                icon: "trash",
                title: "Complete Removal",
                description: "Deleting the app removes all associated data"
            )
        }
        .padding(.vertical, 8)
    }
}

struct InfoRow: View {
    let icon: String
    let title: String
    let description: String
    
    var body: some View {
        HStack(alignment: .top, spacing: 12) {
            Image(systemName: icon)
                .foregroundColor(.blue)
                .frame(width: 20)
            
            VStack(alignment: .leading, spacing: 2) {
                Text(title)
                    .font(.subheadline)
                    .fontWeight(.medium)
                
                Text(description)
                    .font(.caption)
                    .foregroundColor(.secondary)
                    .fixedSize(horizontal: false, vertical: true)
            }
        }
    }
}

struct ComplianceStatusView: View {
    @State private var complianceStatus: ComplianceStatus?
    
    var body: some View {
        VStack(alignment: .leading, spacing: 12) {
            HStack {
                Image(systemName: complianceStatus?.isCompliant == true ? "checkmark.seal.fill" : "exclamationmark.triangle.fill")
                    .foregroundColor(complianceStatus?.isCompliant == true ? .green : .orange)
                
                Text("App Store Compliance")
                    .font(.headline)
                
                Spacer()
            }
            
            if let status = complianceStatus {
                Text(status.summary)
                    .font(.body)
                    .foregroundColor(.secondary)
                
                if !status.issues.isEmpty {
                    ForEach(status.issues, id: \.self) { issue in
                        Label(issue, systemImage: "exclamationmark.circle")
                            .font(.caption)
                            .foregroundColor(.orange)
                    }
                }
                
                Text("Last checked: \(status.lastChecked.formatted(date: .abbreviated, time: .shortened))")
                    .font(.caption2)
                    .foregroundColor(.secondary)
            }
        }
        .onAppear {
            complianceStatus = AppStoreCompliance.validateCompliance()
        }
    }
}

struct DataExportView: View {
    let historicalData: [PredictionHistoryEntry]?
    let trendAnalysis: TrendAnalysis?
    
    @State private var exportData: String = ""
    @State private var isExporting = false
    @Environment(\.dismiss) private var dismiss
    
    init(historicalData: [PredictionHistoryEntry]? = nil, trendAnalysis: TrendAnalysis? = nil) {
        self.historicalData = historicalData
        self.trendAnalysis = trendAnalysis
    }
    
    var body: some View {
        NavigationStack {
            ScrollView {
                VStack(alignment: .leading, spacing: 16) {
                    Text("Export Your Data")
                        .font(.title2)
                        .fontWeight(.bold)
                    
                    Text("Below is all the personal data stored by this app. You can copy this information for your records.")
                        .font(.body)
                        .foregroundColor(.secondary)
                    
                    if isExporting {
                        ProgressView("Preparing data export...")
                            .padding()
                    } else {
                        Text(exportData)
                            .font(.system(.caption, design: .monospaced))
                            // textSelection not available in watchOS
                            .padding()
                            .background(Color.gray.opacity(0.1))
                            .cornerRadius(8)
                            .frame(maxHeight: 300)
                        
                        // UIPasteboard not available in watchOS
                        // Could implement sharing via other means if needed
                        Text("Data ready for export")
                            .font(.caption)
                            .foregroundColor(.secondary)
                    }
                }
                .padding()
            }
            .navigationTitle("Data Export")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .cancellationAction) {
                    Button("Done") {
                        dismiss()
                    }
                }
            }
        }
        .onAppear {
            generateDataExport()
        }
    }
    
    private func generateDataExport() {
        isExporting = true
        
        DispatchQueue.global(qos: .userInitiated).async {
            let data = collectUserData()
            
            DispatchQueue.main.async {
                self.exportData = data
                self.isExporting = false
            }
        }
    }
    
    private func collectUserData() -> String {
        var data = "PERSONAL DATA EXPORT\n"
        data += "Generated: \(Date().formatted(date: .complete, time: .complete))\n"
        data += "App: Life Clock\n"
        data += "Version: 1.0\n\n"
        
        // Include historical data if provided
        if let historicalData = historicalData, !historicalData.isEmpty {
            data += "HISTORICAL PREDICTION DATA:\n"
            data += "Total entries: \(historicalData.count)\n"
            
            for entry in historicalData.sorted(by: { $0.timestamp < $1.timestamp }) {
                data += "\nDate: \(entry.timestamp.formatted(date: .complete, time: .complete))\n"
                data += "Life Expectancy: \(String(format: "%.2f", entry.lifeExpectancy)) years\n"
                
                if let heartRate = entry.avgHeartRate {
                    data += "Average Heart Rate: \(String(format: "%.1f", heartRate)) bpm\n"
                }
                if let steps = entry.dailySteps {
                    data += "Daily Steps: \(String(format: "%.0f", steps))\n"
                }
                if let sleep = entry.sleepHours {
                    data += "Sleep Hours: \(String(format: "%.1f", sleep)) hours\n"
                }
                if let exercise = entry.exerciseMinutes {
                    data += "Exercise Minutes: \(String(format: "%.0f", exercise)) minutes\n"
                }
                if let factors = entry.contributingFactors, !factors.isEmpty {
                    data += "Contributing Factors: \(factors.joined(separator: ", "))\n"
                }
            }
            data += "\n"
        }
        
        // Include trend analysis if provided
        if let analysis = trendAnalysis {
            data += "TREND ANALYSIS:\n"
            data += "Average Life Expectancy: \(String(format: "%.2f", analysis.averageLifeExpectancy)) years\n"
            data += "Trend Direction: \(analysis.trendDirection.rawValue)\n"
            data += "Total Change: \(String(format: "%.2f", analysis.totalChange)) years\n"
            
            if let bestDay = analysis.bestDay {
                data += "Best Day: \(bestDay.timestamp.formatted(date: .complete, time: .complete)) - \(String(format: "%.2f", bestDay.lifeExpectancy)) years\n"
            }
            if let worstDay = analysis.worstDay {
                data += "Worst Day: \(worstDay.timestamp.formatted(date: .complete, time: .complete)) - \(String(format: "%.2f", worstDay.lifeExpectancy)) years\n"
            }
            
            if !analysis.contributingFactors.isEmpty {
                data += "Contributing Factors:\n"
                for factor in analysis.contributingFactors {
                    data += "  - \(factor.name): \(factor.description) (Impact: \(String(format: "%.2f", factor.impact)))\n"
                }
            }
            data += "\n"
        }
        
        data += "DISCLAIMER ACCEPTANCE:\n"
        if let accepted = UserDefaults.standard.object(forKey: "hasAcceptedMedicalDisclaimer") as? Bool,
           let acceptanceDate = UserDefaults.standard.object(forKey: "disclaimerAcceptanceDate") as? Date {
            data += "Accepted: \(accepted)\n"
            data += "Date: \(acceptanceDate.formatted(date: .complete, time: .complete))\n"
        } else {
            data += "No disclaimer acceptance record\n"
        }
        
        data += "\nUSER PREFERENCES:\n"
        // Add any user preferences stored in UserDefaults
        let domain = Bundle.main.bundleIdentifier!
        let defaults = UserDefaults.standard.persistentDomain(forName: domain) ?? [:]
        
        for (key, value) in defaults {
            if !key.contains("disclaimer") { // Already covered above
                data += "\(key): \(value)\n"
            }
        }
        
        data += "\nDISCLAIMER:\n"
        data += "This export contains all personal data stored by the app.\n"
        data += "No data is stored on external servers.\n"
        data += "This information is generated from your device only.\n"
        
        return data
    }
}

#Preview {
    PrivacySettingsView()
}